/**
 * @brief   Adam's Disk Sonic Screwdriver - a File and Disk Manager
 * @author  Thomas Cherryhomes
 * @email   thom dot cherryhomes at gmail dot com
 * @license gpl v. 3, see LICENSE for details.
 * @verbose Directory display
 */

#include <eos.h>
#include <msx.h>
#include <smartkeys.h>
#include <conio.h>
#include "directory.h"
#include "state.h"
#include "rleunpack.h"
#include "eos_filesystem.h"
#include "cpm_filesystem.h"
#include "buffer.h"
#include "globals.h"
#include "input.h"
#include "cursor.h"

// RLE encoded pixel data to go to VDP address 0x0000
static const unsigned char directory_pixels[] = {
  0x8e, 0x00, 0x02, 0x03, 0x07, 0x85, 0x00, 0x03, 0x01, 0x03, 0x07, 0xe4,
  0x00, 0x04, 0xf0, 0x07, 0x03, 0x01, 0xff, 0x00, 0x02, 0x00, 0x07, 0x97,
  0xf0, 0xff, 0x00, 0xe9, 0x00, 0x98, 0xf0, 0xff, 0x00, 0xd9, 0x00, 0x01,
  0x80, 0x87, 0xc0, 0x88, 0x00, 0x98, 0xf0, 0xff, 0x00, 0xd9, 0x00, 0x88,
  0xc0, 0x88, 0x00, 0x98, 0xf0, 0xff, 0x00, 0xd9, 0x00, 0x88, 0xc0, 0x88,
  0x00, 0x98, 0xf0, 0xff, 0x00, 0xd9, 0x00, 0x88, 0xc0, 0x88, 0x00, 0x98,
  0xf0, 0xff, 0x00, 0xd9, 0x00, 0x88, 0xc0, 0x88, 0x00, 0x98, 0xf0, 0xff,
  0x00, 0xd9, 0x00, 0x88, 0xc0, 0x88, 0x00, 0x98, 0xf0, 0xff, 0x00, 0xd9,
  0x00, 0x88, 0xc0, 0x88, 0x00, 0x98, 0xf0, 0xff, 0x00, 0xd9, 0x00, 0x88,
  0xc0, 0x88, 0x00, 0x98, 0xf0, 0xff, 0x00, 0xd9, 0x00, 0x88, 0xc0, 0x88,
  0x00, 0x98, 0xf0, 0xff, 0x00, 0xd9, 0x00, 0x88, 0xc0, 0x88, 0x00, 0x98,
  0xf0, 0xff, 0x00, 0xd9, 0x00, 0x88, 0xc0, 0x88, 0x00, 0x98, 0xf0, 0xff,
  0x00, 0xd9, 0x00, 0x88, 0xc0, 0x88, 0x00, 0x98, 0xf0, 0xff, 0x00, 0xd9,
  0x00, 0x88, 0xc0, 0x88, 0x00, 0x98, 0xf0, 0xff, 0x00, 0xd9, 0x00, 0x88,
  0xc0, 0x88, 0x00, 0x98, 0xf0, 0xff, 0x00, 0xd9, 0x00, 0x88, 0xc0, 0x88,
  0x00, 0x98, 0xf0, 0xff, 0x00, 0xd9, 0x00, 0x88, 0xc0, 0x88, 0x00, 0x98,
  0xf0, 0xff, 0x00, 0xd9, 0x00, 0x88, 0xc0, 0x88, 0x00, 0x96, 0xf0, 0x02,
  0x07, 0x03, 0xff, 0x00, 0xd9, 0x00, 0x87, 0xc0, 0x01, 0x80, 0x88, 0x00,
  0x04, 0xf0, 0xf0, 0x07, 0x03, 0x84, 0x00, 0x02, 0x07, 0x03, 0xff, 0x00,
  0xc6, 0x00, 0x01, 0x01, 0x85, 0x00, 0x01, 0x01, 0x85, 0x00, 0x01, 0x01,
  0x85, 0x00, 0x01, 0x01, 0xff, 0x00, 0xff, 0x00, 0xff, 0x00, 0xff, 0x00,
  0xff, 0x00, 0xff, 0x00, 0xa4, 0x00
};

// RLE encoded color data to go to VDP address 0x2000
static const unsigned char directory_colors[] = {
  0x8e, 0x17, 0x02, 0xe7, 0xe7, 0x85, 0x17, 0x03, 0xf7, 0xfe, 0xfe, 0x84,
  0x17, 0x84, 0x1f, 0x84, 0x17, 0x84, 0x1f, 0x84, 0x17, 0x84, 0x1f, 0x84,
  0x17, 0x84, 0x1f, 0x84, 0x17, 0x84, 0x1f, 0x84, 0x17, 0x84, 0x1f, 0x84,
  0x17, 0x84, 0x1f, 0x84, 0x17, 0x84, 0x1f, 0x84, 0x17, 0x84, 0x1f, 0x84,
  0x17, 0x84, 0x1f, 0x84, 0x17, 0x84, 0x1f, 0x84, 0x17, 0x84, 0x1f, 0x84,
  0x17, 0x01, 0xf7, 0x83, 0x7f, 0xff, 0x17, 0x03, 0x17, 0x17, 0x71, 0x86,
  0xe1, 0x88, 0x1e, 0x88, 0xef, 0xe8, 0x1f, 0xff, 0x17, 0x01, 0x17, 0x88,
  0xe1, 0x88, 0x1e, 0x88, 0xef, 0xff, 0x1f, 0xd9, 0x1f, 0x88, 0xf7, 0x88,
  0x17, 0x88, 0xe1, 0x88, 0x1e, 0x88, 0xef, 0x88, 0x1f, 0xe0, 0x17, 0x88,
  0x1f, 0xe0, 0x17, 0x88, 0x1f, 0x88, 0xf7, 0x88, 0x17, 0x88, 0xe1, 0x88,
  0x1e, 0x88, 0xef, 0xff, 0x1f, 0xd9, 0x1f, 0x88, 0xf7, 0x88, 0x17, 0x88,
  0xe1, 0x88, 0x1e, 0x88, 0xef, 0x88, 0x1f, 0xe0, 0x17, 0x88, 0x1f, 0xe0,
  0x17, 0x88, 0x1f, 0x88, 0xf7, 0x88, 0x17, 0x88, 0xe1, 0x88, 0x1e, 0x88,
  0xef, 0xff, 0x1f, 0xd9, 0x1f, 0x88, 0xf7, 0x88, 0x17, 0x88, 0xe1, 0x88,
  0x1e, 0x88, 0xef, 0x88, 0x1f, 0xe0, 0x17, 0x88, 0x1f, 0xe0, 0x17, 0x88,
  0x1f, 0x88, 0xf7, 0x88, 0x17, 0x88, 0xe1, 0x88, 0x1e, 0x88, 0xef, 0xff,
  0x1f, 0xd9, 0x1f, 0x88, 0xf7, 0x88, 0x17, 0x88, 0xe1, 0x88, 0x1e, 0x88,
  0xef, 0x88, 0x1f, 0xe0, 0x17, 0x88, 0x1f, 0xe0, 0x17, 0x88, 0x1f, 0x88,
  0xf7, 0x88, 0x17, 0x88, 0xe1, 0x88, 0x1e, 0x88, 0xef, 0xff, 0x1f, 0xd9,
  0x1f, 0x88, 0xf7, 0x88, 0x17, 0x88, 0xe1, 0x88, 0x1e, 0x88, 0xef, 0x88,
  0x1f, 0xe0, 0x17, 0x88, 0x1f, 0xe0, 0x17, 0x88, 0x1f, 0x88, 0xf7, 0x88,
  0x17, 0x88, 0xe1, 0x88, 0x1e, 0x88, 0xef, 0xff, 0x1f, 0xd9, 0x1f, 0x88,
  0xf7, 0x88, 0x17, 0x88, 0xe1, 0x88, 0x1e, 0x88, 0xef, 0x88, 0x1f, 0xe0,
  0x17, 0x88, 0x1f, 0xe0, 0x17, 0x88, 0x1f, 0x88, 0xf7, 0x88, 0x17, 0x88,
  0xe1, 0x88, 0x1e, 0x88, 0xef, 0xff, 0x1f, 0xd9, 0x1f, 0x88, 0xf7, 0x88,
  0x17, 0x88, 0xe1, 0x88, 0x1e, 0x88, 0xef, 0x88, 0x1f, 0xe0, 0x17, 0x88,
  0x1f, 0xe0, 0x17, 0x88, 0x1f, 0x88, 0xf7, 0x88, 0x17, 0x88, 0xe1, 0x88,
  0x1e, 0x88, 0xef, 0xff, 0x1f, 0xd9, 0x1f, 0x88, 0xf7, 0x88, 0x17, 0x88,
  0xe1, 0x88, 0x1e, 0x88, 0xef, 0x88, 0x1f, 0xe0, 0x17, 0x88, 0x1f, 0xe0,
  0x17, 0x88, 0x1f, 0x88, 0xf7, 0x88, 0x17, 0x88, 0xe1, 0x88, 0x1e, 0x88,
  0xef, 0xff, 0x1f, 0xd9, 0x1f, 0x88, 0xf7, 0x88, 0x17, 0x88, 0xe1, 0x88,
  0x1e, 0x86, 0xef, 0x02, 0xfe, 0xfe, 0xff, 0x1f, 0xd9, 0x1f, 0x88, 0xf7,
  0x88, 0x17, 0x02, 0xe1, 0xe1, 0x84, 0x1e, 0x7f, 0x11, 0x11, 0xe1, 0xe1,
  0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e, 0x1e,
  0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e, 0x1e,
  0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e, 0x1e,
  0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e, 0x1e,
  0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e, 0x1e,
  0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e, 0x1e,
  0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e, 0x1e,
  0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e, 0x1e,
  0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e, 0x1e,
  0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e, 0x1e,
  0x11, 0x11, 0x1e, 0x57, 0x1e, 0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e,
  0x1e, 0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e,
  0x1e, 0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e,
  0x1e, 0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e,
  0x1e, 0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e,
  0x1e, 0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e, 0x1e, 0x11, 0x11, 0x1e,
  0x1e, 0x11, 0x11, 0x1e, 0x1e, 0x11, 0x71, 0x1e, 0x1e, 0x11, 0x11, 0x1e,
  0x7e, 0x17, 0x17, 0x1e, 0x1e, 0x11, 0x71, 0x84, 0x17, 0x02, 0x1e, 0x7e,
  0xff, 0x17, 0xff, 0x17, 0xff, 0x17, 0xff, 0x17, 0xff, 0x17, 0xff, 0x17,
  0xa4, 0x17
};

void directory_bkg(void)
{
  msx_color(1,15,7);
  msx_set_border(7);
  RLEUnpack(0x0000,directory_pixels,MODE2_MAX);
  RLEUnpack(0x2000,directory_colors,MODE2_MAX);
}

void directory_bkg_remove_bluelines(void)
{
  msx_vfill(MODE2_ATTR+0x0300+32,0x1F,200);
  msx_vfill(MODE2_ATTR+0x0500+32,0x1F,200);
  msx_vfill(MODE2_ATTR+0x0700+32,0x1F,200);
  msx_vfill(MODE2_ATTR+0x0900+32,0x1F,200);
  msx_vfill(MODE2_ATTR+0x0B00+32,0x1F,200);
  msx_vfill(MODE2_ATTR+0x0D00+32,0x1F,200);
  msx_vfill(MODE2_ATTR+0x0F00+32,0x1F,200);
  msx_vfill(MODE2_ATTR+0x1100+32,0x1F,200);
}

// Print entry into slot on screen at pos

void directory_display_entry(char *s, unsigned char pos)
{
  int r = pos >> 1;
  int c = pos % 2;
  int y = r + 3;
  int x = c ? 17 : 4;
  int b = r % 2 ? 0x0F : 0x07;
  
  msx_color(1,b,7);
  gotoxy(x,y);
  
  cputs(s);
}

// clear entries to slots

void directory_clear_entries_to_slots(void)
{
  for (unsigned char i=0;i<32; i++)
    entries_to_slots[i]=-1;
}

void directory_update_cursor(unsigned char pos)
{
  int r = pos >> 1;
  int c = pos % 2;
  int y = r + 3;
  int x = c ? 16 : 3;

  selector(1);
  selector_pos(x,y);
}

// Move directory cursor on screen

void directory_up(void)
{
  if (current_pos)
    current_pos -= 2;

  if (current_pos<0)
    current_pos = 0;

  directory_update_cursor(current_pos);
}

void directory_right(void)
{
  if (!(current_pos % 2))
    current_pos++;

  if (entries_to_slots[current_pos]<0)
    current_pos--;
  
  directory_update_cursor(current_pos);
}

void directory_down(void)
{
  int p=current_pos+2;
  
  if (p<0)
    return;
  else if (p>29)
    return;

  if (entries_to_slots[p]<0)
    return;
  else
    current_pos = p;

  directory_update_cursor(current_pos);
}

void directory_left(void)
{
  if ((current_pos % 2))
    current_pos--;

  directory_update_cursor(current_pos);
}

// Display current directory page in Directory (EOS)

void directory_display_eos(void)
{
  unsigned short rlen=0;
  unsigned char err=0, r=0;
  DirectoryEntry *d = (DirectoryEntry *)buffer;
  unsigned char i=1, slot=0;
  char vn[12]={0,0,0,0,0,0,0,0,0,0,0,0};
  char vt;
  
  r = directory_read_eos(current_device,buffer,BUFFER_SIZE,&err,&rlen);

  // Put volume name in tab
  msx_color(1,15,7);
  directory_filename(d[0],vn,&vt);
  gotoxy(3,1);
  cprintf("%s%c",vn,vt);

  // Populate file directory, ignoring locked, system, and deleted files
  while (!entry_is_blocks_left(d[i]))
    {
      if (!wildcard_mode)
	{
	  if (entry_is_locked(d[i]))
	    {
	      i++;
	      continue;
	    }
	  else if (entry_is_system_file(d[i]))
	    {
	      i++;
	      continue;
	    }
	  else if (entry_is_deleted(d[i]))
	    {
	      i++;
	      continue;
	    }
	}

      char fn[12]={0,0,0,0,0,0,0,0,0,0,0,0};
      char ft;
      directory_filename(d[i],fn,&ft);
      directory_display_entry(fn,slot);
      entries_to_slots[slot]=i;
      slot++;
      i++;
    }

  directory_update_cursor(current_pos);
}

void directory_display_cpm(void)
{
  // This is currently a testing harness

  CpmDirectoryEntry *d = (CpmDirectoryEntry *)buffer;
  unsigned char entry=0, slot=0;
  
  // put CP/M in TAB
  msx_color(1,15,7);
  gotoxy(3,1);
  cputs(" CP/M VOLUME");

  current_device=0x05;
  directory_read_cpm(current_device);
  
  while (entry<64)
    {
      if (d[entry].user == 0xE5) // Empty entry
	{
	  entry++;
	  continue;
	}
      else if (d[entry].extent_number > 0) // file extent, skip.
	{
	  entry++;
	  continue;
	}
      else if (valid_cpm_filename_character(d[entry].filename[0]))
	{
	  char fn[14]={0,0,0,0,0,0,0,0,0,0,0,0,0,0};

	  memcpy(&fn[0],d[entry].filename,8);
	  memcpy(&fn[8],d[entry].extension,3);

	  directory_display_entry(fn,slot);

	  slot++;
	  entry++;
	}
    }
}

void directory(void)
{
  directory_bkg();
  buffer_fill(0x00);
  directory_clear_entries_to_slots();
  
  switch (current_filesystem)
    {
    case EOS:
      directory_display_eos();
      break;
    case CPM:
      directory_display_cpm();
      break;
    }

  state=MENU_MAIN;
}
